-- ============================================================================
-- TWINBEAM Asset Tagging & IT Solutions - Complete Database Setup
-- ============================================================================

-- Create database
CREATE DATABASE IF NOT EXISTS twinbeam CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;

USE twinbeam;

-- ============================================================================
-- CORE TABLES
-- ============================================================================

-- 1. Contacts table - For contact form submissions
CREATE TABLE IF NOT EXISTS contacts (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    email VARCHAR(255) NOT NULL,
    subject VARCHAR(255),
    message TEXT,
    phone VARCHAR(20) DEFAULT NULL,
    company VARCHAR(255) DEFAULT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    reply TEXT DEFAULT NULL,
    replied_by VARCHAR(100) DEFAULT NULL,
    replied_at DATETIME DEFAULT NULL,
    status ENUM('new', 'replied', 'closed') DEFAULT 'new',
    INDEX idx_created_at (created_at),
    INDEX idx_email (email),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Stores contact form submissions';

-- 2. Quote requests table - For service quote requests
CREATE TABLE IF NOT EXISTS quote_requests (
    id INT AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    email VARCHAR(255) NOT NULL,
    phone VARCHAR(20) NOT NULL,
    company VARCHAR(255) NOT NULL,
    service_type ENUM('asset_tagging', 'asset_engraving', 'computers', 'networking', 'office_supplies', 'other') NOT NULL,
    number_of_assets INT DEFAULT NULL,
    project_description TEXT,
    preferred_start_date DATE DEFAULT NULL,
    budget_range VARCHAR(100) DEFAULT NULL,
    status ENUM('pending', 'quoted', 'accepted', 'rejected', 'completed') DEFAULT 'pending',
    quote_amount DECIMAL(10, 2) DEFAULT NULL,
    quote_details TEXT DEFAULT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    quoted_at DATETIME DEFAULT NULL,
    quoted_by VARCHAR(100) DEFAULT NULL,
    INDEX idx_status (status),
    INDEX idx_service_type (service_type),
    INDEX idx_created_at (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Stores service quote requests from potential clients';

-- 3. Projects table - Track ongoing projects
CREATE TABLE IF NOT EXISTS projects (
    id INT AUTO_INCREMENT PRIMARY KEY,
    client_name VARCHAR(255) NOT NULL,
    client_email VARCHAR(255) NOT NULL,
    client_phone VARCHAR(20),
    project_name VARCHAR(255) NOT NULL,
    project_type ENUM('asset_tagging', 'asset_engraving', 'it_supply', 'networking', 'consultation', 'audit') NOT NULL,
    start_date DATE,
    end_date DATE,
    status ENUM('planning', 'in_progress', 'on_hold', 'completed', 'cancelled') DEFAULT 'planning',
    total_assets INT DEFAULT NULL,
    tagged_assets INT DEFAULT 0,
    project_value DECIMAL(10, 2),
    notes TEXT,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_status (status),
    INDEX idx_project_type (project_type),
    INDEX idx_client_email (client_email)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Track asset tagging and IT projects';

-- 4. Asset inventory table - Track tagged assets
CREATE TABLE IF NOT EXISTS asset_inventory (
    id INT AUTO_INCREMENT PRIMARY KEY,
    project_id INT,
    asset_tag_number VARCHAR(100) NOT NULL UNIQUE,
    asset_type VARCHAR(100),
    asset_description TEXT,
    asset_brand VARCHAR(100),
    asset_model VARCHAR(100),
    serial_number VARCHAR(100),
    location VARCHAR(255),
    department VARCHAR(100),
    tag_type ENUM('barcode', 'qr_code', 'rfid', 'engraved') DEFAULT 'barcode',
    condition_on_tagging ENUM('excellent', 'good', 'fair', 'poor') DEFAULT 'good',
    tagged_date DATE,
    tagged_by VARCHAR(100),
    notes TEXT,
    photo_url VARCHAR(255),
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (project_id) REFERENCES projects(id) ON DELETE SET NULL,
    INDEX idx_asset_tag (asset_tag_number),
    INDEX idx_project_id (project_id),
    INDEX idx_location (location)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Inventory of all tagged assets';

-- 5. Clients table - Customer database
CREATE TABLE IF NOT EXISTS clients (
    id INT AUTO_INCREMENT PRIMARY KEY,
    company_name VARCHAR(255) NOT NULL,
    contact_person VARCHAR(255),
    email VARCHAR(255) NOT NULL,
    phone VARCHAR(20),
    alternative_phone VARCHAR(20),
    industry ENUM('healthcare', 'education', 'government', 'corporate', 'manufacturing', 'ngo', 'other') DEFAULT 'corporate',
    address TEXT,
    city VARCHAR(100),
    country VARCHAR(100) DEFAULT 'Kenya',
    website VARCHAR(255),
    tax_pin VARCHAR(50),
    status ENUM('active', 'inactive', 'prospect') DEFAULT 'prospect',
    total_projects INT DEFAULT 0,
    total_revenue DECIMAL(12, 2) DEFAULT 0,
    notes TEXT,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_email (email),
    INDEX idx_company_name (company_name),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Client/customer database';

-- 6. Products table - IT equipment and supplies
CREATE TABLE IF NOT EXISTS products (
    id INT AUTO_INCREMENT PRIMARY KEY,
    product_code VARCHAR(50) UNIQUE,
    product_name VARCHAR(255) NOT NULL,
    category ENUM('computers', 'monitors', 'peripherals', 'networking', 'office_supplies', 'asset_tags', 'other') NOT NULL,
    brand VARCHAR(100),
    model VARCHAR(100),
    description TEXT,
    unit_price DECIMAL(10, 2),
    stock_quantity INT DEFAULT 0,
    reorder_level INT DEFAULT 10,
    supplier VARCHAR(255),
    warranty_period VARCHAR(50),
    product_image VARCHAR(255),
    status ENUM('active', 'discontinued', 'out_of_stock') DEFAULT 'active',
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_category (category),
    INDEX idx_product_code (product_code),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Product catalog for IT equipment and supplies';

-- 7. Orders table - Track product orders
CREATE TABLE IF NOT EXISTS orders (
    id INT AUTO_INCREMENT PRIMARY KEY,
    order_number VARCHAR(50) UNIQUE NOT NULL,
    client_id INT,
    order_date DATE NOT NULL,
    total_amount DECIMAL(12, 2) NOT NULL,
    tax_amount DECIMAL(10, 2) DEFAULT 0,
    discount_amount DECIMAL(10, 2) DEFAULT 0,
    grand_total DECIMAL(12, 2) NOT NULL,
    payment_status ENUM('pending', 'partial', 'paid', 'cancelled') DEFAULT 'pending',
    delivery_status ENUM('pending', 'processing', 'shipped', 'delivered', 'cancelled') DEFAULT 'pending',
    delivery_date DATE,
    notes TEXT,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (client_id) REFERENCES clients(id) ON DELETE SET NULL,
    INDEX idx_order_number (order_number),
    INDEX idx_client_id (client_id),
    INDEX idx_payment_status (payment_status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Customer orders';

-- 8. Order items table
CREATE TABLE IF NOT EXISTS order_items (
    id INT AUTO_INCREMENT PRIMARY KEY,
    order_id INT NOT NULL,
    product_id INT,
    product_name VARCHAR(255) NOT NULL,
    quantity INT NOT NULL,
    unit_price DECIMAL(10, 2) NOT NULL,
    total_price DECIMAL(10, 2) NOT NULL,
    FOREIGN KEY (order_id) REFERENCES orders(id) ON DELETE CASCADE,
    FOREIGN KEY (product_id) REFERENCES products(id) ON DELETE SET NULL,
    INDEX idx_order_id (order_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Line items for orders';

-- 9. Services table - Service catalog
CREATE TABLE IF NOT EXISTS services (
    id INT AUTO_INCREMENT PRIMARY KEY,
    service_name VARCHAR(255) NOT NULL,
    service_category ENUM('asset_tagging', 'asset_engraving', 'asset_audit', 'it_installation', 'networking', 'consultation', 'maintenance') NOT NULL,
    description TEXT,
    base_price DECIMAL(10, 2),
    pricing_unit VARCHAR(50) COMMENT 'per asset, per hour, per project, etc.',
    features TEXT COMMENT 'JSON or comma-separated list',
    service_image VARCHAR(255),
    is_active BOOLEAN DEFAULT TRUE,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_category (service_category)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Service offerings catalog';

-- 10. Testimonials table
CREATE TABLE IF NOT EXISTS testimonials (
    id INT AUTO_INCREMENT PRIMARY KEY,
    client_name VARCHAR(255) NOT NULL,
    company VARCHAR(255),
    position VARCHAR(100),
    testimonial_text TEXT NOT NULL,
    rating INT DEFAULT 5 CHECK (rating >= 1 AND rating <= 5),
    client_photo VARCHAR(255),
    is_featured BOOLEAN DEFAULT FALSE,
    is_published BOOLEAN DEFAULT FALSE,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_published (is_published),
    INDEX idx_featured (is_featured)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Client testimonials and reviews';

-- 11. Users/Admin table
CREATE TABLE IF NOT EXISTS users (
    id INT AUTO_INCREMENT PRIMARY KEY,
    username VARCHAR(100) NOT NULL UNIQUE,
    password VARCHAR(255) NOT NULL COMMENT 'Hashed password',
    full_name VARCHAR(255),
    email VARCHAR(255) NOT NULL,
    phone VARCHAR(20),
    role ENUM('admin', 'manager', 'technician', 'sales') DEFAULT 'technician',
    status ENUM('active', 'inactive') DEFAULT 'active',
    last_login DATETIME,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_username (username),
    INDEX idx_role (role)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='System users and administrators';

-- 12. Blog posts table
CREATE TABLE IF NOT EXISTS blog_posts (
    id INT AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(255) NOT NULL,
    slug VARCHAR(255) UNIQUE NOT NULL,
    content TEXT NOT NULL,
    excerpt TEXT,
    featured_image VARCHAR(255),
    category ENUM('asset_management', 'technology', 'case_studies', 'tips', 'news', 'general') DEFAULT 'general',
    author_id INT,
    tags VARCHAR(255) COMMENT 'Comma-separated tags',
    views INT DEFAULT 0,
    is_published BOOLEAN DEFAULT FALSE,
    published_at DATETIME,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (author_id) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_slug (slug),
    INDEX idx_published (is_published),
    INDEX idx_category (category)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Blog and news articles';

-- 13. Newsletters/Subscribers table
CREATE TABLE IF NOT EXISTS newsletter_subscribers (
    id INT AUTO_INCREMENT PRIMARY KEY,
    email VARCHAR(255) NOT NULL UNIQUE,
    name VARCHAR(255),
    company VARCHAR(255),
    subscription_date DATETIME DEFAULT CURRENT_TIMESTAMP,
    status ENUM('active', 'unsubscribed') DEFAULT 'active',
    unsubscribe_token VARCHAR(100),
    INDEX idx_email (email),
    INDEX idx_status (status)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Newsletter subscribers';

-- 14. FAQ categories table
CREATE TABLE IF NOT EXISTS faq_categories (
    id INT AUTO_INCREMENT PRIMARY KEY,
    category_name VARCHAR(100) NOT NULL,
    display_order INT DEFAULT 0,
    is_active BOOLEAN DEFAULT TRUE
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='FAQ category organization';

-- 15. FAQs table
CREATE TABLE IF NOT EXISTS faqs (
    id INT AUTO_INCREMENT PRIMARY KEY,
    category_id INT,
    question TEXT NOT NULL,
    answer TEXT NOT NULL,
    display_order INT DEFAULT 0,
    views INT DEFAULT 0,
    is_published BOOLEAN DEFAULT TRUE,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (category_id) REFERENCES faq_categories(id) ON DELETE SET NULL,
    INDEX idx_category (category_id),
    INDEX idx_published (is_published)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Frequently asked questions';

-- 16. Site settings table
CREATE TABLE IF NOT EXISTS site_settings (
    id INT AUTO_INCREMENT PRIMARY KEY,
    setting_key VARCHAR(100) UNIQUE NOT NULL,
    setting_value TEXT,
    setting_type ENUM('text', 'number', 'boolean', 'json') DEFAULT 'text',
    description VARCHAR(255),
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='Website configuration settings';

-- 17. Activity log table
CREATE TABLE IF NOT EXISTS activity_log (
    id INT AUTO_INCREMENT PRIMARY KEY,
    user_id INT,
    action VARCHAR(100) NOT NULL,
    entity_type VARCHAR(50) COMMENT 'project, order, client, etc.',
    entity_id INT,
    description TEXT,
    ip_address VARCHAR(45),
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_user_id (user_id),
    INDEX idx_entity (entity_type, entity_id),
    INDEX idx_created_at (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
COMMENT='System activity tracking';

-- ============================================================================
-- INSERT SAMPLE DATA
-- ============================================================================

-- Insert default admin user (password: admin123)
INSERT INTO users (username, password, full_name, email, role) VALUES
('admin', '240be518fabd2724ddb6f04eeb1da5967448d7e831c08c8fa822809f74c720a9', 'System Administrator', 'admin@twinbeam.co.ke', 'admin'),
('manager', '240be518fabd2724ddb6f04eeb1da5967448d7e831c08c8fa822809f74c720a9', 'Operations Manager', 'manager@twinbeam.co.ke', 'manager')
ON DUPLICATE KEY UPDATE username=username;

-- Insert FAQ categories
INSERT INTO faq_categories (category_name, display_order, is_active) VALUES
('General Questions', 1, TRUE),
('Services', 2, TRUE),
('Technical', 3, TRUE),
('Implementation', 4, TRUE)
ON DUPLICATE KEY UPDATE category_name=category_name;

-- Insert sample FAQs
INSERT INTO faqs (category_id, question, answer, display_order, is_published) VALUES
(1, 'What is asset tagging?', 'Asset tagging is the process of labeling physical assets with unique identification tags that help organizations track, manage, and secure their equipment. These tags can include barcodes, QR codes, RFID chips, or engraved serial numbers.', 1, TRUE),
(1, 'Why is asset tagging important?', 'Asset tagging improves asset visibility, accountability, and control. It helps reduce asset loss and theft, track asset location, improve audit accuracy, support preventive maintenance, and enhance asset lifecycle management.', 2, TRUE),
(2, 'Do you offer asset engraving services?', 'Yes. We provide professional laser and mechanical engraving services for permanent asset identification. Engraving is ideal for high-value or theft-prone assets where permanent marking is required.', 1, TRUE),
(3, 'How durable are your asset tags?', 'Our asset tags are designed to withstand harsh conditions including heat, cold, moisture, chemicals, and abrasion. They are suitable for indoor, outdoor, and industrial environments.', 1, TRUE),
(4, 'How long does an asset tagging project take?', 'Project timelines depend on the number of assets. Small offices: 1-3 days, Medium organizations: 1-2 weeks, Large enterprises: project-based scheduling.', 1, TRUE)
ON DUPLICATE KEY UPDATE question=question;

-- Insert sample services
INSERT INTO services (service_name, service_category, description, base_price, pricing_unit, is_active) VALUES
('Asset Tagging - Barcode', 'asset_tagging', 'Professional barcode asset tagging with durable labels', 50.00, 'per asset', TRUE),
('Asset Tagging - QR Code', 'asset_tagging', 'QR code asset labeling for mobile scanning', 60.00, 'per asset', TRUE),
('Asset Tagging - RFID', 'asset_tagging', 'RFID asset tracking solution', 150.00, 'per asset', TRUE),
('Laser Engraving', 'asset_engraving', 'Permanent laser engraving on equipment', 200.00, 'per asset', TRUE),
('Asset Audit & Verification', 'asset_audit', 'Complete physical asset verification and reconciliation', 5000.00, 'per project', TRUE),
('Network Installation', 'networking', 'Complete network infrastructure setup', 10000.00, 'per project', TRUE)
ON DUPLICATE KEY UPDATE service_name=service_name;

-- Insert sample products
INSERT INTO products (product_code, product_name, category, brand, unit_price, stock_quantity, status) VALUES
('LAP-HP-001', 'HP ProBook 450 G8', 'computers', 'HP', 65000.00, 15, 'active'),
('MON-DELL-001', 'Dell 24" LED Monitor', 'monitors', 'Dell', 18000.00, 25, 'active'),
('NET-CISCO-001', 'Cisco 24-Port Switch', 'networking', 'Cisco', 45000.00, 8, 'active'),
('TAG-BAR-001', 'Barcode Asset Labels (100pcs)', 'asset_tags', 'Generic', 2500.00, 100, 'active'),
('TAG-QR-001', 'QR Code Asset Labels (100pcs)', 'asset_tags', 'Generic', 3000.00, 80, 'active')
ON DUPLICATE KEY UPDATE product_code=product_code;

-- Insert sample testimonials
INSERT INTO testimonials (client_name, company, position, testimonial_text, rating, is_published, is_featured) VALUES
('John Kimani', 'Nairobi Hospital', 'IT Manager', 'TWINBEAM provided excellent asset tagging services. Their team was professional and completed the project on time. We can now track all our medical equipment efficiently.', 5, TRUE, TRUE),
('Mary Wanjiru', 'University of Nairobi', 'Procurement Officer', 'The asset engraving service was top-notch. All our computers and projectors are now permanently marked. Great work!', 5, TRUE, FALSE),
('David Omondi', 'Kenya Airways', 'Asset Manager', 'Professional service delivery and excellent customer support. Highly recommended for any organization looking to improve their asset management.', 5, TRUE, TRUE)
ON DUPLICATE KEY UPDATE client_name=client_name;

-- Insert site settings
INSERT INTO site_settings (setting_key, setting_value, setting_type, description) VALUES
('company_name', 'TWINBEAM Asset Solutions', 'text', 'Company name'),
('company_email', 'info@twinbeam.co.ke', 'text', 'Main contact email'),
('company_phone', '+254 700 123 456', 'text', 'Main contact phone'),
('company_address', 'Nairobi, Kenya', 'text', 'Company address'),
('site_tagline', 'Professional Asset Tagging & IT Solutions', 'text', 'Website tagline'),
('enable_quotes', '1', 'boolean', 'Enable quote request system'),
('enable_blog', '1', 'boolean', 'Enable blog feature')
ON DUPLICATE KEY UPDATE setting_key=setting_key;

-- ============================================================================
-- CREATE VIEWS
-- ============================================================================

-- View for project statistics
CREATE OR REPLACE VIEW v_project_stats AS
SELECT 
    project_type,
    COUNT(*) as total_projects,
    SUM(CASE WHEN status = 'completed' THEN 1 ELSE 0 END) as completed,
    SUM(CASE WHEN status = 'in_progress' THEN 1 ELSE 0 END) as ongoing,
    SUM(project_value) as total_revenue
FROM projects
GROUP BY project_type;

-- View for client summary
CREATE OR REPLACE VIEW v_client_summary AS
SELECT 
    c.*,
    COUNT(p.id) as project_count,
    SUM(p.project_value) as total_spent
FROM clients c
LEFT JOIN projects p ON c.id = p.client_name
GROUP BY c.id;

-- View for low stock products
CREATE OR REPLACE VIEW v_low_stock_products AS
SELECT * FROM products
WHERE stock_quantity <= reorder_level AND status = 'active';

-- View for pending quotes
CREATE OR REPLACE VIEW v_pending_quotes AS
SELECT * FROM quote_requests
WHERE status = 'pending'
ORDER BY created_at DESC;

-- ============================================================================
-- CREATE STORED PROCEDURES
-- ============================================================================

-- Procedure to generate order number
DELIMITER //
CREATE PROCEDURE IF NOT EXISTS sp_generate_order_number()
BEGIN
    DECLARE new_number VARCHAR(50);
    DECLARE year_prefix VARCHAR(4);
    DECLARE seq_num INT;
    
    SET year_prefix = YEAR(NOW());
    
    SELECT COALESCE(MAX(CAST(SUBSTRING(order_number, 6) AS UNSIGNED)), 0) + 1 
    INTO seq_num
    FROM orders
    WHERE order_number LIKE CONCAT(year_prefix, '%');
    
    SET new_number = CONCAT(year_prefix, LPAD(seq_num, 5, '0'));
    
    SELECT new_number;
END//
DELIMITER ;

-- ============================================================================
-- CREATE TRIGGERS
-- ============================================================================

-- Update client project count after project insert
DELIMITER //
CREATE TRIGGER IF NOT EXISTS trg_after_project_insert
AFTER INSERT ON projects
FOR EACH ROW
BEGIN
    UPDATE clients 
    SET total_projects = total_projects + 1,
        total_revenue = total_revenue + COALESCE(NEW.project_value, 0)
    WHERE company_name = NEW.client_name;
END//
DELIMITER ;

-- Update product stock after order
DELIMITER //
CREATE TRIGGER IF NOT EXISTS trg_after_order_item_insert
AFTER INSERT ON order_items
FOR EACH ROW
BEGIN
    UPDATE products 
    SET stock_quantity = stock_quantity - NEW.quantity
    WHERE id = NEW.product_id;
END//
DELIMITER ;

-- ============================================================================
-- SETUP COMPLETE
-- ============================================================================

SELECT 'Database setup completed successfully!' as Status,
       'TWINBEAM Asset Solutions' as Company,
       DATABASE() as Database_Name,
       (SELECT COUNT(*) FROM information_schema.tables WHERE table_schema = DATABASE()) as Tables_Created;

-- Show table summary
SELECT 
    table_name as 'Table Name',
    table_rows as 'Rows',
    ROUND(((data_length + index_length) / 1024 / 1024), 2) as 'Size (MB)'
FROM information_schema.tables
WHERE table_schema = DATABASE()
ORDER BY table_name;