<?php
// subscribe.php - Central subscription handler
session_start();

// Database configuration
$host = 'localhost';
$dbname = 'twinbeam_db';  // Make sure this matches your database name
$username = 'root';
$password = '';  // Add your password if you have one

// Create connection
try {
    $pdo = new PDO("mysql:host=$host;dbname=$dbname", $username, $password);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch(PDOException $e) {
    die("Connection failed: " . $e->getMessage());
}

// Function to sanitize input
function sanitize($data) {
    return htmlspecialchars(strip_tags(trim($data)));
}

// Function to validate email
function validateEmail($email) {
    return filter_var($email, FILTER_VALIDATE_EMAIL);
}

// Function to validate phone number (Kenyan format)
function validatePhone($phone) {
    // Remove spaces and special characters
    $phone = preg_replace('/[^0-9]/', '', $phone);
    
    // Check if it matches Kenyan phone format (0712345678 or 712345678 or +254712345678)
    if (preg_match('/^(?:\+254|254|0)?([17]\d{8})$/', $phone, $matches)) {
        return '0' . $matches[1]; // Normalize to 0712345678 format
    }
    return false;
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $response = array();
    
    // Get subscription type
    $subscription_type = sanitize($_POST['subscription_type'] ?? 'general');
    
    // Get form data
    $service_number = sanitize($_POST['service_number'] ?? '');
    $phone = sanitize($_POST['phone'] ?? '');
    $email = sanitize($_POST['email'] ?? '');
    $name = sanitize($_POST['name'] ?? '');
    
    // Validation
    $errors = array();
    
    if (empty($service_number)) {
        $errors[] = "Service number is required";
    }
    
    if (empty($phone)) {
        $errors[] = "Phone number is required";
    } else {
        $validated_phone = validatePhone($phone);
        if (!$validated_phone) {
            $errors[] = "Invalid phone number format";
        } else {
            $phone = $validated_phone;
        }
    }
    
    if (!empty($email) && !validateEmail($email)) {
        $errors[] = "Invalid email address";
    }
    
    if (!empty($errors)) {
        $response['status'] = 'error';
        $response['message'] = implode(', ', $errors);
        echo json_encode($response);
        exit;
    }
    
    // Check if already subscribed
    $stmt = $pdo->prepare("SELECT id FROM subscribers WHERE service_number = ? OR phone = ?");
    $stmt->execute([$service_number, $phone]);
    
    if ($stmt->fetch()) {
        // Update existing subscription
        $stmt = $pdo->prepare("UPDATE subscribers SET 
            name = COALESCE(NULLIF(?, ''), name),
            email = COALESCE(NULLIF(?, ''), email),
            subscription_type = ?,
            updated_at = NOW()
            WHERE service_number = ? OR phone = ?");
        
        $stmt->execute([$name, $email, $subscription_type, $service_number, $phone]);
        
        $response['status'] = 'success';
        $response['message'] = 'Your subscription has been updated successfully!';
    } else {
        // Insert new subscription
        $stmt = $pdo->prepare("INSERT INTO subscribers 
            (service_number, name, phone, email, subscription_type, created_at) 
            VALUES (?, ?, ?, ?, ?, NOW())");
        
        $stmt->execute([$service_number, $name, $phone, $email, $subscription_type]);
        
        $response['status'] = 'success';
        $response['message'] = 'You have been subscribed successfully!';
        
        // Send confirmation SMS (integrate with SMS gateway)
        sendConfirmationSMS($phone, $name);
        
        // Send confirmation email if provided
        if (!empty($email)) {
            sendConfirmationEmail($email, $name);
        }
    }
    
    echo json_encode($response);
    exit;
}

// Function to send confirmation SMS
function sendConfirmationSMS($phone, $name) {
    // Integrate with Africa's Talking or any SMS gateway
    // Example implementation:
    
    $message = "Dear " . ($name ?: 'Member') . ", you have successfully subscribed to DEFMIS notifications. You will receive updates via SMS. To unsubscribe, reply STOP.";
    
    // Africa's Talking implementation example
    /*
    require_once 'path/to/AfricasTalkingGateway.php';
    
    $username = "your_username";
    $apiKey = "your_api_key";
    $gateway = new AfricasTalkingGateway($username, $apiKey);
    
    try {
        $results = $gateway->sendMessage($phone, $message);
    } catch (Exception $e) {
        error_log("SMS Error: " . $e->getMessage());
    }
    */
    
    // For now, just log it
    error_log("SMS to $phone: $message");
}

// Function to send confirmation email
function sendConfirmationEmail($email, $name) {
    $to = $email;
    $subject = "DEFMIS Subscription Confirmation";
    $message = "
    <html>
    <head>
        <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: linear-gradient(135deg, #1e3c72 0%, #2a5298 100%); color: white; padding: 20px; text-align: center; }
            .content { padding: 20px; background: #f9f9f9; }
            .footer { padding: 20px; text-align: center; font-size: 12px; color: #666; }
        </style>
    </head>
    <body>
        <div class='container'>
            <div class='header'>
                <h1>Welcome to DEFMIS Notifications</h1>
            </div>
            <div class='content'>
                <p>Dear " . ($name ?: 'Member') . ",</p>
                <p>Thank you for subscribing to DEFMIS notifications. You will now receive:</p>
                <ul>
                    <li>Important announcements</li>
                    <li>Event updates</li>
                    <li>Press releases</li>
                    <li>Healthcare tips and reminders</li>
                </ul>
                <p>If you have any questions, please contact us at info@defmis.org or call 0798 999 444.</p>
            </div>
            <div class='footer'>
                <p>Defence Forces Medical Insurance Scheme<br>
                19th Floor Nyayo House, P.O Box 21710-00100, Nairobi<br>
                www.defmis.org</p>
            </div>
        </div>
    </body>
    </html>
    ";
    
    $headers = "MIME-Version: 1.0" . "\r\n";
    $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
    $headers .= "From: DEFMIS <noreply@defmis.org>" . "\r\n";
    
    mail($to, $subject, $message, $headers);
}

// Unsubscribe handler
if (isset($_GET['unsubscribe']) && isset($_GET['phone'])) {
    $phone = validatePhone($_GET['phone']);
    
    if ($phone) {
        $stmt = $pdo->prepare("UPDATE subscribers SET status = 'unsubscribed', updated_at = NOW() WHERE phone = ?");
        $stmt->execute([$phone]);
        
        echo "You have been unsubscribed successfully.";
    }
    exit;
}
?>