<?php
/**
 * Database Setup Checker
 * Run this file to check if your database is properly configured
 * Access: http://localhost/TWINBEAM/setup_check.php
 */

error_reporting(E_ALL);
ini_set('display_errors', 1);

// Database configuration
$host = 'localhost';
$username = 'root';
$password = '';
$dbname = 'twinbeam_db';

// Check results
$checks = [];

// 1. Check MySQL connection
try {
    $conn = new mysqli($host, $username, $password);
    if ($conn->connect_error) {
        $checks['mysql_connection'] = [
            'status' => 'error',
            'message' => 'Cannot connect to MySQL: ' . $conn->connect_error
        ];
    } else {
        $checks['mysql_connection'] = [
            'status' => 'success',
            'message' => 'MySQL connection successful'
        ];
    }
} catch (Exception $e) {
    $checks['mysql_connection'] = [
        'status' => 'error',
        'message' => 'MySQL connection error: ' . $e->getMessage()
    ];
}

// 2. Check if database exists
if ($checks['mysql_connection']['status'] === 'success') {
    $result = $conn->query("SHOW DATABASES LIKE '$dbname'");
    if ($result->num_rows > 0) {
        $checks['database_exists'] = [
            'status' => 'success',
            'message' => "Database '$dbname' exists"
        ];
        
        // Select the database
        $conn->select_db($dbname);
        
        // 3. Check required tables
        $required_tables = [
            'contacts',
            'appointments',
            'users',
            'hospitals',
            'subscribers',
            'notification_logs',
            'admin_users',
            'events',
            'announcement_posts',
            'press_releases'
        ];
        
        $existing_tables = [];
        $missing_tables = [];
        
        foreach ($required_tables as $table) {
            $result = $conn->query("SHOW TABLES LIKE '$table'");
            if ($result->num_rows > 0) {
                $existing_tables[] = $table;
            } else {
                $missing_tables[] = $table;
            }
        }
        
        if (empty($missing_tables)) {
            $checks['tables'] = [
                'status' => 'success',
                'message' => 'All required tables exist (' . count($existing_tables) . ' tables)',
                'details' => $existing_tables
            ];
        } else {
            $checks['tables'] = [
                'status' => 'warning',
                'message' => count($existing_tables) . ' tables exist, ' . count($missing_tables) . ' missing',
                'existing' => $existing_tables,
                'missing' => $missing_tables
            ];
        }
        
        // 4. Check table data
        $table_counts = [];
        foreach ($existing_tables as $table) {
            $result = $conn->query("SELECT COUNT(*) as count FROM $table");
            $row = $result->fetch_assoc();
            $table_counts[$table] = $row['count'];
        }
        
        $checks['data'] = [
            'status' => 'info',
            'message' => 'Table data counts',
            'details' => $table_counts
        ];
        
    } else {
        $checks['database_exists'] = [
            'status' => 'error',
            'message' => "Database '$dbname' does not exist! Please create it first."
        ];
    }
}

$conn->close();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Database Setup Checker - DEFMIS</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <style>
        body {
            background: #f8f9fa;
            padding: 40px 0;
        }
        .check-card {
            margin-bottom: 20px;
            border-left: 4px solid;
        }
        .check-card.success { border-left-color: #28a745; }
        .check-card.error { border-left-color: #dc3545; }
        .check-card.warning { border-left-color: #ffc107; }
        .check-card.info { border-left-color: #17a2b8; }
        .status-icon {
            font-size: 1.5rem;
            margin-right: 10px;
        }
        .code-box {
            background: #2d2d2d;
            color: #f8f8f2;
            padding: 20px;
            border-radius: 5px;
            font-family: 'Courier New', monospace;
            font-size: 0.9rem;
            overflow-x: auto;
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="row justify-content-center">
            <div class="col-lg-10">
                <div class="card shadow-lg">
                    <div class="card-header bg-primary text-white">
                        <h3 class="mb-0">
                            <i class="fas fa-database me-2"></i>
                            DEFMIS Database Setup Checker
                        </h3>
                    </div>
                    <div class="card-body p-4">
                        
                        <!-- MySQL Connection Check -->
                        <div class="card check-card <?php echo $checks['mysql_connection']['status']; ?>">
                            <div class="card-body">
                                <h5>
                                    <span class="status-icon">
                                        <?php if ($checks['mysql_connection']['status'] === 'success'): ?>
                                            <i class="fas fa-check-circle text-success"></i>
                                        <?php else: ?>
                                            <i class="fas fa-times-circle text-danger"></i>
                                        <?php endif; ?>
                                    </span>
                                    MySQL Connection
                                </h5>
                                <p class="mb-0"><?php echo $checks['mysql_connection']['message']; ?></p>
                            </div>
                        </div>

                        <!-- Database Exists Check -->
                        <?php if (isset($checks['database_exists'])): ?>
                        <div class="card check-card <?php echo $checks['database_exists']['status']; ?>">
                            <div class="card-body">
                                <h5>
                                    <span class="status-icon">
                                        <?php if ($checks['database_exists']['status'] === 'success'): ?>
                                            <i class="fas fa-check-circle text-success"></i>
                                        <?php else: ?>
                                            <i class="fas fa-times-circle text-danger"></i>
                                        <?php endif; ?>
                                    </span>
                                    Database Exists
                                </h5>
                                <p class="mb-0"><?php echo $checks['database_exists']['message']; ?></p>
                                
                                <?php if ($checks['database_exists']['status'] === 'error'): ?>
                                <div class="alert alert-danger mt-3">
                                    <h6><i class="fas fa-exclamation-triangle me-2"></i>Quick Fix:</h6>
                                    <p>Run this SQL command in phpMyAdmin:</p>
                                    <div class="code-box">
                                        CREATE DATABASE defmis_db CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
                                    </div>
                                    <p class="mt-2 mb-0"><strong>OR</strong> run the complete setup script: <code>setup_database.sql</code></p>
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        <?php endif; ?>

                        <!-- Tables Check -->
                        <?php if (isset($checks['tables'])): ?>
                        <div class="card check-card <?php echo $checks['tables']['status']; ?>">
                            <div class="card-body">
                                <h5>
                                    <span class="status-icon">
                                        <?php if ($checks['tables']['status'] === 'success'): ?>
                                            <i class="fas fa-check-circle text-success"></i>
                                        <?php elseif ($checks['tables']['status'] === 'warning'): ?>
                                            <i class="fas fa-exclamation-triangle text-warning"></i>
                                        <?php endif; ?>
                                    </span>
                                    Required Tables
                                </h5>
                                <p><?php echo $checks['tables']['message']; ?></p>
                                
                                <?php if (!empty($checks['tables']['existing'])): ?>
                                <div class="mb-3">
                                    <h6 class="text-success"><i class="fas fa-check me-2"></i>Existing Tables:</h6>
                                    <div class="row">
                                        <?php foreach ($checks['tables']['existing'] as $table): ?>
                                        <div class="col-md-6">
                                            <span class="badge bg-success mb-1">✓ <?php echo $table; ?></span>
                                        </div>
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                                <?php endif; ?>
                                
                                <?php if (!empty($checks['tables']['missing'])): ?>
                                <div class="alert alert-warning">
                                    <h6><i class="fas fa-exclamation-triangle me-2"></i>Missing Tables:</h6>
                                    <div class="row">
                                        <?php foreach ($checks['tables']['missing'] as $table): ?>
                                        <div class="col-md-6">
                                            <span class="badge bg-warning text-dark mb-1">✗ <?php echo $table; ?></span>
                                        </div>
                                        <?php endforeach; ?>
                                    </div>
                                    <hr>
                                    <p class="mb-0"><strong>Fix:</strong> Run the complete setup script: <code>setup_database.sql</code> in phpMyAdmin</p>
                                </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        <?php endif; ?>

                        <!-- Data Check -->
                        <?php if (isset($checks['data'])): ?>
                        <div class="card check-card info">
                            <div class="card-body">
                                <h5>
                                    <span class="status-icon">
                                        <i class="fas fa-info-circle text-info"></i>
                                    </span>
                                    Table Data
                                </h5>
                                <table class="table table-sm">
                                    <thead>
                                        <tr>
                                            <th>Table Name</th>
                                            <th>Records</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($checks['data']['details'] as $table => $count): ?>
                                        <tr>
                                            <td><code><?php echo $table; ?></code></td>
                                            <td><span class="badge bg-primary"><?php echo $count; ?></span></td>
                                        </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                        <?php endif; ?>

                        <!-- Action Buttons -->
                        <div class="card">
                            <div class="card-body text-center">
                                <h5 class="mb-4">Next Steps</h5>
                                <div class="d-grid gap-2 d-md-flex justify-content-md-center">
                                    <a href="http://localhost/phpmyadmin" target="_blank" class="btn btn-primary">
                                        <i class="fas fa-database me-2"></i>Open phpMyAdmin
                                    </a>
                                    <a href="index.php" class="btn btn-success">
                                        <i class="fas fa-home me-2"></i>Go to Website
                                    </a>
                                    <button onclick="location.reload()" class="btn btn-info">
                                        <i class="fas fa-sync me-2"></i>Re-check
                                    </button>
                                </div>
                            </div>
                        </div>

                    </div>
                </div>

                <!-- Configuration Info -->
                <div class="card mt-4">
                    <div class="card-header bg-secondary text-white">
                        <h5 class="mb-0"><i class="fas fa-cog me-2"></i>Current Configuration</h5>
                    </div>
                    <div class="card-body">
                        <table class="table table-bordered">
                            <tr>
                                <th>Database Host:</th>
                                <td><code><?php echo $host; ?></code></td>
                            </tr>
                            <tr>
                                <th>Database Name:</th>
                                <td><code><?php echo $dbname; ?></code></td>
                            </tr>
                            <tr>
                                <th>Database User:</th>
                                <td><code><?php echo $username; ?></code></td>
                            </tr>
                            <tr>
                                <th>PHP Version:</th>
                                <td><code><?php echo phpversion(); ?></code></td>
                            </tr>
                        </table>
                        <p class="mb-0 text-muted">
                            <small><i class="fas fa-info-circle me-1"></i>
                            To change these settings, edit: <code>includes/db_config.php</code></small>
                        </p>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>